# #!/system/bin/sh
# SPDX-License-Identifier: MIT
# Copyright (C) 2022 Leica Geosystems AG

# This script is not deployed as part of the system partition.
# It's deployed directly in AOSP archive together with partition images.
# During OTA generation, it's copied from AOSP archive into OTA archive.
# This way when update is executed by update script (sysupdate.sh), it can be
# extracted from OTA and executed at the very beginning of the update process
# (before system partition is flashed).

# This script takes care of recovery partition update. Functional recovery
# partition was not deployed in FW 1.0.0 and was not planned to be used
# beforehand - thus corresponding support is missing from sysupdate.sh.
# Execution of this script, i.e. pre_ota_script.sh is a fallback mechansim
# emebdded in sysupdate.sh before 1.0.0 release foreseen to extend its
# functionality.


set -o pipefail
set -e
set -x

RECOVERY_PART=/dev/block/bootdevice/by-name/recovery
OTA_PATH=/cache/processor_package.zip
NUM_VER_BYTES=6
# Recovery image version is embedded as 6 first bytes in "asciiz product name"
# field of the image header (6 bytes at offset 48)
# Version of the recovery image is configured in AOSP's top-level core/Makefile
OD_CMD="od -A n -t a -j 48 -N $NUM_VER_BYTES"

# Update of the recovery partition is risky - there's no backup partition.
#
# Only execute the update if the version of the recovery image embedded in its
# header is lower than the version of recovery image in OTA package.

log_msg() {
    log -p I -t "_sysupdate_" "pre_ota_script.sh: $@"
    echo "pre_ota_script.sh: $@"
}

is_recovery_mode_active() {
    local active_rec=$(getprop ro.kgf.active.recovery)
    log_msg "Active recovery: $active_rec"

    [ -n "$active_rec" ]
}

get_rec_ver_from_ota_package() {
    # When executed in context of service, unzip exits with error 141 and od
    # with error 2 - silence those and any other errors. Output of this
    # function is later verified for correctness
    (set +o pipefail; unzip -c -p "$1" recovery.img | ${OD_CMD} | sed 's/ //g')
}

get_rec_ver_cur_installed() {
    ${OD_CMD} "${1}" | sed 's/ //g'
}

# Checks if passed argument is a 6-charcater representation of a decimal number
is_6_digit_number() {
    [ "$1" != "$(printf "%0${NUM_VER_BYTES}d" $(($1)))" ]
}

is_recovery_update_needed() {
    local cur_ver="$1"
    local ota_ver="$2"

    # OTA version does not represent 6 digit number - do not flash it
    is_6_digit_number "$ota_ver" && return 1

    # Recovery version not set (that's the case for FW 1.0.0, od tool reports 0
    # bytes as 'nul'). Check explicitly even though it would be also covered
    # by subsequent check.
    [ "$cur_ver" == "nulnulnulnulnulnul" ] && return 0

    # Curent version does not represent 6 digit number
    is_6_digit_number "$cur_ver" && return 0

    [ $ota_ver -gt $cur_ver  ]
}

execute_recovery_update() {
    log_msg "Executing update of recovery partition"

    if unzip -c -p "$1" recovery.img | \
        busybox-android dd of="$RECOVERY_PART" oflag=direct bs=4096 2>/dev/null; then
        log_msg "Update successful"
        return 0
    fi

    log_msg "ERROR: update of recovery partition failed!"

    return 1
}

log_msg "Executing pre_ota_script.sh (<$0> <$@>)"

if is_recovery_mode_active; then
    log_msg "Running from within recovery mode - skipping recovery partition"\
        "update"
    exit 0
fi

if [ ! -f $OTA_PATH ]; then
    log_msg "$OTA_PATH does not exist"
    exit 1
fi

if ! REC_VER_OTA=$(get_rec_ver_from_ota_package $OTA_PATH); then
    log_msg "Failed to extract version from $OTA_PATH"
    exit 1
fi

if ! REC_VER_CUR=$(get_rec_ver_cur_installed $RECOVERY_PART); then
    log_msg "Failed to get version from $RECOVERY_PATH"
    exit 1
fi

log_msg "Recovery image. Currently installed: $REC_VER_CUR, in OTA:"\
    "$REC_VER_OTA"

if ! is_recovery_update_needed "$REC_VER_CUR" "$REC_VER_OTA"; then
    log_msg "Recovery update not needed"
    exit 0
fi

execute_recovery_update "$OTA_PATH"
